/*
 * ProfileSequence.cc
 *
 * This class allows a number of manipulations to `ProfileSet's  stored in a file.
 *
 */

#ifdef USE_GL
#include <gl/gl.h>
#ifndef LINUX
// We might have doublebuffering available.  Try it...
#define MULTI_BUFFER 1
#endif
#else
#include <vogl.h>
#include <vodevice.h>
#include <X11/Xutil.h>
#endif

#include <cmath>
#include <cstdlib>
#include <ctime>
#include <cassert>

#include "ProfileSequence.h"

#include "PCAclass.h"
#include "MeanProfile.h"
#include "NagVector.h"
#include "Tracker.h"

namespace ReadingPeopleTracker
{

// definition and initialisation of static member variables

realno ProfileSequence::NO_DEVIATIONS = 1.6;
//  FIXME: Maybe ok to have these former configuration variable constant.  Or is it?
//  Configuration.register_real("DISPLAY_STD_DEVIATIONS", 1.6,
//  		      0.0, 4.0,
//  		      &no_deviations, false,
//  		      "ProfileSequence", 
//  		      "The maximum number of std deviations from 
//  the mean shape to the displayed animated shape.");

unsigned int ProfileSequence::NO_SHAPE_MODES = 10;
//  FIXME: Maybe ok to have these former configuration variable constant.  Or is it?
//  Configuration.register_int("NO_SHAPE_MODES", 10,
//  		     &NO_SHAPE_MODES, false,
//  		     "ProfileSequence",
//  		     "The number of shape parameters used to 
//  represent shapes. This is used if you specify a model file and a 
//  sequence file and specify the SHOW_SHAPES option.");

unsigned int ProfileSequence::WAIT_TIME = 50;
//  FIXME: Maybe ok to have these former configuration variable constant.  Or is it?
//  Configuration.register_int("WAIT_TIME", 50,
//  		     &WAIT_TIME, false,
//  		     "ProfileSequence",
//  		     "A delay factor (in msec) used when displaying a set of shapes");


ProfileSequence::ProfileSequence(char *name, char *model_name)
    : weights()
{
    if (name != NULL) 
    {
	strncpy(filename, name, 99);
	ifstream data_in(filename);
	data_in >> no_ctrl_pnts;
	Profile::NO_CONTROL_POINTS = no_ctrl_pnts;
	data_in.close();
    }
    else
	filename[0] = '\0';

    if (model_name != NULL)
    {
	pca = new PCAclass(model_name);
	Profile::NO_CONTROL_POINTS = pca->data_size / 2;

	if (Profile::draw_arrow == true)
	    Profile::NO_CONTROL_POINTS--;

	mean = new MeanProfile();
	mean->PCA_flag = 0;
	
	pca->mean.copy(*mean);
	mean->get_size();
	pca->weights.copy(weights);
    }
    else 
    {
	mean = new MeanProfile();
	mean->PCA_flag = 0;
	pca = NULL;
    }
    setup_H_metric(mean->get_data_size());

    // initialise pscale
    pscale = 1.0;
}

ProfileSequence::~ProfileSequence()
{
    delete mean;
    if (pca != NULL)
	delete pca;
}

void ProfileSequence::calculate_mean()
{
    ifstream data_in(filename);
    ProfileSet curr_profiles;
    mean->reset_mean();
    data_in >> Profile::NO_CONTROL_POINTS;
    while (data_in.peek() != EOF)
    {
	data_in >> curr_profiles;
	
	// remove profiles that are not ok
	curr_profiles.keep_ok();
	
	// transform to "V" space
	curr_profiles.transform(*get_H_root());
	
	mean->update(curr_profiles);
	curr_profiles.destroy_all();
    }
    data_in.close();
//   setup_H_metric(*mean);
//   get_H_root()->multiply(*mean);
}

unsigned int ProfileSequence::count_profiles()
{
    ifstream data_in(filename);
    data_in >> Profile::NO_CONTROL_POINTS;
    unsigned int no_items = 0;
    ProfileSet curr_profiles;
    while (data_in.peek() != EOF)
    {
	data_in >> curr_profiles;
	no_items += curr_profiles.no_items;
	curr_profiles.destroy_all();
    }
    data_in.close();
    return no_items;
}

void ProfileSequence::convert_to_matrix(NagMatrix &training_matrix,
					realno use_fraction)
{
    normal_random();	// initialise seed if necessary
    unsigned short seed_buff[3];
    unsigned short *curr_seed = seed48(seed_buff);
    
    // copy over the current seed into the buffer
    memcpy(&seed_buff[0], curr_seed, sizeof(unsigned short[3]));
    
    seed48(seed_buff);
    ifstream data_in(filename);
    data_in >> Profile::NO_CONTROL_POINTS;
    cinfo << " counting training set " << endl;
    unsigned int no_items = 0;
    ProfileSet curr_profiles;
    while (data_in.peek() != EOF)
    {
	data_in >> curr_profiles;
	for (ListNode<Profile> *curr = curr_profiles.first;
	     curr != NULL; curr = curr->next)
	    if (drand48() <= use_fraction)
		no_items++;
	curr_profiles.destroy_all();
    }
    data_in.close();
    cinfo << " found " << no_items << " training items " << endl;
    Profile dummy;
    training_matrix.reconstruct(dummy.get_data_size(), no_items);
    ifstream data_in2(filename);
    data_in2 >> Profile::NO_CONTROL_POINTS;
    int indx = 0;
    
    seed48(seed_buff);		// reset the seed to buffer
    while (data_in2.peek() != EOF)
    {
	data_in2 >> curr_profiles;
	curr_profiles.transform(*get_H_root());
	for (ListNode<Profile> *curr = curr_profiles.first;
	     curr != NULL; curr = curr->next)
	    if (drand48() <= use_fraction)
		curr->dat->copy(training_matrix.column(indx++));
	
	curr_profiles.destroy_all();
    }   
}  

void ProfileSequence::do_analysis()
{
    calculate_mean();

    if (pca != NULL)
	delete pca;

    pca = new PCAclass(mean->get_data_size());
    setup_weights();
    weights.copy(pca->weights);
    //  mean->recenter(weights.get_data());
    mean->direction = Point2(0,0);
    // don't need to copy over mean
    // it will be recalculated !
    //   mean->NagVector::copy(pca->mean);
    ifstream data_in(filename);
    ProfileSet *curr_profiles = new ProfileSet();
    ProfileSet *old_profiles = NULL;
    
    ListNode<Profile> *curr;
    data_in >> Profile::NO_CONTROL_POINTS;
    cinfo << " calculating covariance .." << flush;
    bool do_noise_model = false;
    
    while(data_in.peek() != EOF)
    {
	data_in >> *curr_profiles;
	// remove profiles that are not ok
	curr_profiles->keep_ok();
	curr_profiles->transform(*get_H_root());
	
	if (((pca->no_processed) % 10) == 0)
	    cinfo << "." << flush;
	for (curr = curr_profiles->first; curr != NULL; curr = curr->next)
	{
	    curr->dat->direction.normalise();
	    //align_profile(*curr->dat);
	    
	    if (curr->dat->PCA_flag == 0)
		pca->process_cov(*curr->dat);
	   

	    cdebug << " ProfileSequence::do_analysis(): assign a value to PCA_flag??? "
		   << endl;
 
	    if ((old_profiles != NULL) && (old_profiles->first != NULL)
		&& (curr->dat->PCA_flag > 0))
	    {
		ListNode<Profile> *oldp;
		for (oldp = old_profiles->first;
		     (oldp !=  NULL) && (oldp->dat->PCA_flag != curr->dat->PCA_flag);
		     oldp = oldp->next)
		{
		    // nothing
		};
		
		if (oldp != NULL)
		{
		    pca->process_cov(*curr->dat, *oldp->dat);
		    do_noise_model = true;
		}
	    }
	}
	if (old_profiles != NULL)
	{
	    old_profiles->destroy_all();
	    delete old_profiles;
	}
	
	old_profiles = curr_profiles;
	curr_profiles = new ProfileSet();
	
    }
    
    old_profiles->destroy_all();
    delete old_profiles;
    delete curr_profiles;
    
    data_in.close();
    cinfo << endl << pca->no_processed << " items processed " << endl;
    cinfo << " finding modes ..." << endl;
    pca->restore_covariance();
    pca->get_evals();
    pca->mean.NagVector::copy(*mean);
    if (do_noise_model)
	pca->get_noise_model();
}

void ProfileSequence::display_shapes(Point2 shape_origin) 
{
    ifstream data_in(filename);
    ProfileSet curr_profiles;
    data_in >> Profile::NO_CONTROL_POINTS;
    clear(); 
    ListNode<Profile> *curr;
    while (!data_in.eof())
    {
	data_in >> curr_profiles;
	for (curr = curr_profiles.first; curr != NULL; curr = curr->next)
	{
// FIXME: bool Profile::is_ok() commented out!  problem: access to configuration
// FIXME: variables MAX_HEIGHT_TO_WIDTH, MIN_HEIGHT_TO_WIDTH, NO_BLOB_FILTER.
// FIXME:	    if (curr->dat->is_ok())
// FIXME:	    {
		if (pca != NULL) 
		{
		    NagVector shape_parms(NO_SHAPE_MODES);
		    //DONT ALIGN -- ASSUME profiles are in 
		    // model frame
		    // otherwise components are not available!!!
		    align_profile(*curr->dat);
		    pca->map_to_model(NO_SHAPE_MODES, *curr->dat,
				      shape_parms);
		    for (unsigned int i = 0; i < NO_SHAPE_MODES; i++)
			cout << shape_parms[i] << "   ";
		    
		    cout << endl;
		}
		if (pca != NULL)
		    pca->map_to_shape(NO_SHAPE_MODES, *curr->dat, 1e12);
		
		curr->dat->display_shape(shape_origin.x, shape_origin.y);
		//	      sginap(WAIT_TIME);
#ifdef LINUX   // usleep uses microseconds as input
		usleep(long(WAIT_TIME*1000));
#else
		usleep(long((WAIT_TIME/CLK_TCK)*1000000));
#endif
// FIXME:	    }
	}
	RGBcolor(0,0,0);
	clear();
	curr_profiles.destroy_all();
    }
    data_in.close();
}

void ProfileSequence::edit_shapes(Point2 shape_origin, char *file_out_name)
{
    ofstream data_out(file_out_name);
    
    ifstream data_in(filename);
    ProfileSet curr_profiles;
    ProfileSet new_profiles;
    data_in >> Profile::NO_CONTROL_POINTS;
    data_out << Profile::NO_CONTROL_POINTS << " control points " << endl;
    unsigned char input;
    
//    RGBcolor(0,0,0);
//    clear();

    ListNode<Profile> *curr;

    while (!data_in.eof())
    {
	data_in >> curr_profiles;
	for (curr = curr_profiles.first; curr != NULL; curr = curr->next)
	{
	    RGBcolor(0,0,0);
	    clear();
	    curr->dat->display_shape(shape_origin.x, shape_origin.y);
	    gflush();

	    input = getchar();
	    
	    if ((input != 'n') && (input != 'N'))
	    {
		new_profiles.add_a_copy(curr->dat);
		cdebug << ".";
	    }
	    else
		cdebug << "X";
	    
	}
	data_out << new_profiles;
	data_out << "********************" << endl;
	curr_profiles.destroy_all();
	new_profiles.destroy_all();
    }
    data_in.close();
    data_out.close();
    
}

void ProfileSequence::reset_window(realno &ox, realno &oy, int mode)
{
    Int32 x1;
    Int32 y1;
    Int32 dx;
    Int32 dy;
    getorigin(&x1, &y1);
    getsize(&dx, &dy);
    viewport(0, dx-1, 0, dy-1);
    ortho2(-0.5,(dx / pscale) + 0.5,-0.5, (dy /pscale) +0.5);
    ox = (dx / (2.0 * pscale));
    oy = (dy / (2.0 * pscale));
#ifdef MULTI_BUFFER
    doublebuffer();
#endif
    gconfig();
    unsigned char title_str[20];
    sprintf(title_str,"mode %d", mode);
    
#ifdef MULTI_BUFFER
    for (int i = 0; i < 2; i++)
    {
#endif
	viewport(0, dx-1, 0, dy-1);
	RGBcolor(0,0,0);
	clear();
	RGBcolor(255,255,255);
	ortho2(-0.5,dx + 0.5,-0.5, dy +0.5);
	cmov2i(32,32);
	unsigned charstr(title_str);
	ortho2(-0.5,(dx / pscale) + 0.5,-0.5, (dy /pscale) +0.5);
#ifdef MULTI_BUFFER
	swapbuffers();
    }
#endif
}

void ProfileSequence::display_mode(int &mode)
{
    if (mode < 1)
	mode = 1;
    if (mode > (2 * Profile::NO_CONTROL_POINTS))
	mode = 2 * Profile::NO_CONTROL_POINTS;
    
    realno theta = 0;
    realno small_ang = (2 * 3.14159265358979323) / 160;
    realno max = no_deviations * sqrt(pca->get_eval(mode));
    Profile old_profile = Profile(*mean);
    Profile current_profile = Profile(*mean);
    current_profile.get_size();
    current_profile.draw_in_colour = true;
    
    realno ox,oy;
    
    prefsize(300,400);
    foreground();
    winopen("PCA");
#ifdef USE_GL
    RGBmode();
    gconfig();
//    RGBcolor(0,0,0);
//    clear();
#endif
    winconstraints();
    reset_window(ox,oy,mode);

    qdevice(REDRAW);
    qdevice(ESCKEY);
    qdevice(ONEKEY);
    qdevice(TWOKEY);
    qdevice(AKEY);
    
#ifdef USE_GL
    qdevice(UPARROWKEY);
    qdevice(DOWNARROWKEY);
    qdevice(RIGHTARROWKEY);
    qdevice(LEFTARROWKEY);
#endif
    
    short val;
    long dev;
    
    qreset();
    int old_mode = mode;
    while (mode != 0)
    {
	unsigned int count = 0;
	while (qtest() == 0)  // waiting for X / GL events
	{
	    max = no_deviations * sqrt(pca->get_eval(mode));
	    pca->get_mode(mode, max * sin (theta), current_profile);
	    theta += small_ang;
	    if (count > 0)
		old_profile.display_shape(ox,oy,false);
	    current_profile.display_shape(ox,oy,true);
	    old_profile = current_profile;
	    count++;

	    usleep(10000);
	}
	

	dev = qread(&val);

	if (mode != old_mode)
	{
	    old_mode = mode;
	    reset_window(ox, oy, mode);
	}

	switch (dev)
	{
	case REDRAW:
	    reset_window(ox,oy,mode);
	    current_profile.display_shape(ox,oy,true);
#ifdef MULTI_BUFFER
	    swapbuffers();
#endif
	    break;	    

	case AKEY:
	    reset_window(ox,oy,mode);
	    current_profile.display_shape(ox,oy,true);
#ifdef MULTI_BUFFER
	    swapbuffers();
#endif
	    break;

	case ONEKEY:
	    pscale = 1.0;
	    reset_window(ox,oy,mode);
	    break;
	    
	case TWOKEY:
	    pscale = 2.0;
	    reset_window(ox,oy,mode);
	    break;
	    
#ifdef USE_GL
	case UPARROWKEY:
	    pscale *= 1.1;
	    reset_window(ox,oy,mode);
	    while (qread(&val) != UPARROWKEY)
		usleep(2000);
	    break;
	    
	case DOWNARROWKEY:
	    pscale /= 1.1;
	    reset_window(ox,oy,mode);
	    while (qread(&val) != DOWNARROWKEY)
		usleep(2000);
	    break;
	    
	case RIGHTARROWKEY:
	    mode++;
	    if (mode >= pca->data_size)
		mode = pca->data_size  - 1;
	    theta = 0.0;
	    while (qread(&val) != RIGHTARROWKEY)
		usleep(2000);
	    reset_window(ox,oy,mode); 
	    break;
	    
	case LEFTARROWKEY:
	    mode--;
	    if (mode < 1)
		mode = 1;
	    theta = 0.0;
	    while (qread(&val) != LEFTARROWKEY)
		usleep(2000);
	    reset_window(ox,oy,mode); 
	    break;
#endif   // ifdef USE_GL

	case ESCKEY:
	default:
	    break; // and break again, below.
	}

	if (dev == ESCKEY)
	    break;
    }
    qreset();    
}

void ProfileSequence::random_walk(int no_modes)
{
    int i;
    realno theta = 0;
    realno *max = new realno[no_modes];
    int *rnd = new int[no_modes];
    for (i = 0; i < no_modes; i++)
    {
	max[i] = sqrt(pca->get_eval(i+1)) * exp(-0.04 * (i+1));
	rnd[i] = (rand() % 5) + 1;
    }
    Profile old_profile = Profile(*mean);
    Profile current_profile;
    realno ox = mean->width * 2;
    realno oy = mean->height * 2;
    while (theta < 12)
    {
	pca->get_mode(1, no_deviations * max[i] * sin (rnd[0] * theta),
		      current_profile);
	for (i = 1; i < no_modes; i ++)
	    pca->adjust_mode(i+1, no_deviations * max[i] * 
			     sin (rnd[i] * theta),
			     current_profile);
	theta += 0.02;
	old_profile.display_shape(ox,oy,false);
	current_profile.display_shape(ox,oy,true);
	//      sginap(1);
	usleep(long(1000000/CLK_TCK));
	old_profile = current_profile;
    }
    old_profile.display_shape(ox,oy,false);
    delete [] max;
    delete [] rnd;
}


void ProfileSequence::setup_weights()
{
    Profile::NO_CONTROL_POINTS = no_ctrl_pnts;
    // setup weights to be inversely proportional to variance
    weights = NagVector(Profile::NO_CONTROL_POINTS);
    mean->mean_distances.square_entries(mean->mean_distances);
    mean->mean_squares.subtract(mean->mean_distances,
				mean->mean_squares);
    
    realno sum;
    realno total;
    total = 0;
    for (int i = 0; i < Profile::NO_CONTROL_POINTS; i++)
    {
	sum = 0;
	for (int j = 0; j < Profile::NO_CONTROL_POINTS; j++)
	    sum += mean->mean_squares.read(i,j);
	weights[i] = 1.0 / sum;
	total += weights[i];
    }
    weights /= total;
}

/*
  void ProfileSequence::setup_weights()
  {
  ifstream data_in(filename);
  data_in >> Profile::NO_CONTROL_POINTS;
  // setup weights to be inversely proportional to variance
  if (weights == NULL) weights = new realno[Profile::NO_CONTROL_POINTS];
  if (pca != NULL) 
  {
  pca->weights = weights;
  pca->no_weights = Profile::NO_CONTROL_POINTS;
  }
  for (int i = 0; i < Profile::NO_CONTROL_POINTS; i++)
  weights[i] = 0;
  realno total = 0;
  ProfileSet curr_profiles;
  ListNode<Profile> *curr;
  Point2* pspline;
  while(data_in.peek() != EOF)
  {
  data_in >> curr_profiles;
  for (curr = curr_profiles.first; curr != NULL; curr =
  curr->next)
  {
  pspline = &(curr->dat->spline[0]);
  realno ymin = 1e6;int min_y;
  realno ymax = -1e6;int max_y;
  realno xmin = 1e6;int min_x;
  realno xmax = -1e6;int max_x;
  for (i = 0; i < Profile::NO_CONTROL_POINTS; i++)
  {
  if (pspline[i].y < ymin) 
  {
  ymin = pspline[i].y;
  min_y = i;
  }
  if (pspline[i].y > ymax) 
  {
  ymax = pspline[i].y;
  max_y = i;
  }
  if (pspline[i].x < xmin) 
  {
  xmin = pspline[i].x;
  min_x = i;
  }
  if (pspline[i].x > xmax) 
  {
  xmax = pspline[i].x;
  max_x = i;
  }
  }
  weights[min_y]++; weights[max_y]++;
  //weights[min_x]++; weights[max_x]++;
  
  total +=2;
  }
  curr_profiles.destroy_all();
  }
  data_in.close();
  for (i = 0; i < Profile::NO_CONTROL_POINTS; i++)
  weights[i] /= total;
  }
*/

bool ProfileSequence::align_profile(Profile &current)
{
// FIXME: bool Profile::is_ok() commented out!  problem: access to configuration
// FIXME: variables MAX_HEIGHT_TO_WIDTH, MIN_HEIGHT_TO_WIDTH, NO_BLOB_FILTER.
// FIXME:     if (!current.is_ok()) 
// FIXME:     {
// FIXME: 	//abort();
// FIXME: 	return false;
// FIXME:     }
    realno s, theta;
    //current.recenter(weights);
    current.align_to(mean,&current, s, theta, weights.get_data());
    if ((fabs(theta > 0.78)) || (s < 0.5)) 
    {
	//abort();
	return false;
    }
    return true;
}

void ProfileSequence::align(char *file_out)
{
    char *file_out_name = NULL;
    char *tmp_filename = NULL;
    
    if (file_out == NULL) 
    {
	file_out_name = new char[strlen(filename) + strlen(".aligned____")];
	tmp_filename = new char[strlen(filename) + strlen(".temp___")];
	sprintf(file_out_name, "%s.aligned", filename);
	sprintf(tmp_filename, "%s.temp", filename);
    }
    else 
    {
	file_out_name = new char[strlen(file_out)];
	tmp_filename = new char[strlen(file_out) + strlen(".temp___")];
	strcpy(file_out_name, file_out);
	sprintf(tmp_filename,"%s.temp", file_out);
    }
    
    ifstream data_in(filename);
    ofstream data_out(tmp_filename);
    char *command_buffer = 
	new char[strlen(tmp_filename) + strlen(file_out_name) + strlen("mv_____")];
    
    calculate_mean();
    const int MAX_ITERATE = 100;
    for (unsigned int lp = 0; lp < MAX_ITERATE; lp++)
    {
	setup_weights();
	mean->normalise();
	mean->recenter(weights.get_data());
	mean->origin = Point2(0,0);
	data_in >> Profile::NO_CONTROL_POINTS;
	data_out << Profile::NO_CONTROL_POINTS << " control points " << endl;
	ProfileSet curr_profiles;
	ListNode<Profile> *curr, *nxt;
	MeanProfile *new_mean = new MeanProfile();
	new_mean->PCA_flag = 0;
	new_mean->reset_mean();
	while (data_in.peek() != EOF)
	{
	    data_in >> curr_profiles;
	    curr_profiles.transform(*get_H_root());
	    
	    for (curr = curr_profiles.first; curr != NULL; curr = nxt)
	    {
		nxt = curr->next;
		if (!(align_profile(*(curr->dat))))
		    curr_profiles.destroy(curr);
	    }
	    new_mean->update(curr_profiles);
	    
	    curr_profiles.transform(*get_H_inv_root());
	    data_out << curr_profiles;
	    data_out << "******" << endl;
	    curr_profiles.destroy_all();
	}
	new_mean->normalise();
	new_mean->recenter(weights.get_data());
	NagVector mean_change;
	new_mean->NagVector::subtract(*mean, mean_change);
	delete mean;
	mean = new_mean;
	cout << "Change in means = " << mean_change.length2() << endl;
	data_in.close(); data_out.close();
	
	sprintf(command_buffer,"mv %s %s", tmp_filename, file_out_name);
	system(command_buffer);
	
	if (mean_change.length2() < 0.01)
	    break;

	if (lp != (MAX_ITERATE-1))
	{
	    data_in.open(file_out_name);
	    data_out.open(tmp_filename);
	}
    }
    delete [] command_buffer;
    delete [] file_out_name;
    delete [] tmp_filename;
    data_in.close();
    data_out.close();
}

void ProfileSequence::reflect(char *file_out)
{
    char file_out_name[256];
    if (file_out == NULL) 
    {
	strncpy(file_out_name, filename, 255);
	strncat(file_out_name, ".reflect", 255);
    }
    else
	strncpy(file_out_name, file_out, 255);

    ofstream data_out(file_out_name);
    
    ifstream data_in(filename);
    data_in >> Profile::NO_CONTROL_POINTS;
    data_out << Profile::NO_CONTROL_POINTS << " control points " << endl;
    ProfileSet curr_profiles;
    ProfileSet reflected;
    ListNode<Profile> *curr, *nxt;
    while (data_in.peek() != EOF)
    {
	data_in >> curr_profiles;
	curr_profiles.reflect(reflected);
	data_out << reflected;
	data_out << "******" << endl;
	curr_profiles.destroy_all();
	reflected.destroy_all();
    }
    data_in.close();
    data_out.close();
}

void ProfileSequence::track(char *file_out)
{
    char file_out_name[256];
    if (file_out == NULL) 
    {
	strncpy(file_out_name, filename, 255);
	strncat(file_out_name, ".track", 255);
    }
    else
	strncpy(file_out_name, file_out, 255);

    ofstream data_out(file_out_name);
    Tracker tracker;
    ifstream data_in(filename);
    data_in >> Profile::NO_CONTROL_POINTS;
    data_out << Profile::NO_CONTROL_POINTS << " control points " << endl;
    ProfileSet *new_profiles = NULL;
    ProfileSet *old_profiles = NULL;
    if (data_in.peek() != EOF) 
    {
	new_profiles = new ProfileSet(); 
	data_in >> (*new_profiles);
	tracker.process_frame(new_profiles);
    }
    while (data_in.peek() != EOF)
    {
	old_profiles = new_profiles;
	new_profiles = new ProfileSet();
	data_in >> (*new_profiles);
	tracker.process_frame(new_profiles);
	data_out << (*old_profiles);
	data_out << "******" << endl;
	old_profiles->destroy_all();
	delete old_profiles;
    }
    new_profiles->destroy_all();
    delete new_profiles;
    data_in.close();
    data_out.close();
}



void ProfileSequence::halve(char *file_out)
{
    if (pca == NULL)
	do_analysis();
    char *file_out_name1;
    char *file_out_name2;
    if (file_out == NULL)
    {
	int slen = strlen(filename) + strlen(".half.1") + 4;
	file_out_name1 = new char[slen];
	file_out_name2 = new char[slen];
	sprintf(file_out_name1, "%s.half.1", filename);
	sprintf(file_out_name2, "%s.half.2", filename);
    }
    else 
    {
	int slen = strlen(filename) + strlen(".half.1") + 4;
	file_out_name1 = new char[slen];
	file_out_name2 = new char[slen];
	sprintf(file_out_name1,"%s.1", file_out);
	sprintf(file_out_name2,"%s.2", file_out);
    }
    
    ofstream data_out1(file_out_name1);
    ofstream data_out2(file_out_name2);
    
    ifstream data_in(filename);
    
    data_in >> Profile::NO_CONTROL_POINTS;
    data_out1 << Profile::NO_CONTROL_POINTS << " control points " << endl;
    data_out2 << Profile::NO_CONTROL_POINTS << " control points " << endl;
    
    ProfileSet profiles_in;
    ProfileSet profiles_out1;
    ProfileSet profiles_out2;
    
    while (data_in.peek() != EOF)
    {
	data_in >> profiles_in;
	for (ListNode<Profile> *curr = profiles_in.first;
	     curr != NULL; curr = curr->next)
	{
	    realno m1 = pca->map_to_mode(0,*curr->dat);
	    if (m1 >= 0)
		profiles_out1.add_a_copy(curr->dat);
	    else
		profiles_out2.add_a_copy(curr->dat);
	}
	
	data_out1 << profiles_out1 << "******" << endl;
	data_out2 << profiles_out2 << "******" << endl;
	
	profiles_in.destroy_all();
	profiles_out1.destroy_all();
	profiles_out2.destroy_all();
    }
    data_in.close();
    data_out1.close();
    data_out2.close();
}

bool ProfileSequence::get_next_chain(ProfileSet &prf_chain,
				     unsigned int &last_ref,
				     ifstream &data_in)
{
    prf_chain.destroy_all();
    
    ProfileSet tmp_set;
    ListNode<Profile> *curr = NULL;
    
    streampos back_pos;
    
    while ((data_in.peek() != EOF) && (curr == NULL))
    {
	tmp_set.destroy_all();
	back_pos = data_in.tellg();
	data_in >> tmp_set;
	for (curr = tmp_set.first; (curr != NULL) && 
		 (curr->dat->PCA_flag <= last_ref); curr = curr->next);
    }
    
    if (curr == NULL) 
    {
	data_in.close();
	return false;
    }
    
    int curr_ref = curr->dat->PCA_flag;
    
    prf_chain.add_a_copy(curr->dat);
    tmp_set.destroy_all();
    while ((data_in.peek() != EOF) && (curr != NULL))
    {
	data_in >> tmp_set;
	for (curr = tmp_set.first; (curr != NULL) &&
		 (curr->dat->PCA_flag != curr_ref); curr = curr->next);
	if (curr != NULL)
	    prf_chain.add_a_copy(curr->dat);
	tmp_set.destroy_all();
    }
    
    data_in.seekg(back_pos);
    last_ref = curr_ref;
    return true;
}

void ProfileSequence::center(char *file_out)
{
    char file_out_name[256];
    if (file_out == NULL) 
    {
	strncpy(file_out_name, filename, 255);
	strncat(file_out_name, ".centered", 255);
    }
    else
	strncpy(file_out_name, file_out, 255);

    ofstream data_out(file_out_name);
    calculate_mean();
    setup_weights(); 
    
    int curr_ref = -1;
    data_out << Profile::NO_CONTROL_POINTS << " control points " << endl;
    ProfileSet prf_chain;
    
    ifstream data_in(filename);
    
    while (get_next_chain(prf_chain, curr_ref, data_in))
    {
	ListNode<Profile> *curr;
	for (curr = prf_chain.first; curr != NULL; curr = curr->next)
	    curr->dat->recenter(weights.get_data()); 
	
	prf_chain.fit_to_line();
	ProfileSet tmp_set;
	for (curr = prf_chain.first; curr != NULL;
	     curr = curr->next)
	{
	    assert (tmp_set.no_items == 0);
	    tmp_set.add(curr->dat);
	    data_out << tmp_set << "******" << endl;
	    tmp_set.delete_all();
	}
    }
    prf_chain.destroy_all();
    data_in.close();
    data_out.close();
}

} // namespace ReadingPeopleTracker
