/*
 * PCAclass
 * 
 * a class for carrying out Principal Component Analysis on
 * arbitrary data
 *
 */

#include <cmath>
#include <cstring>
#include <cassert>

#include "PCAclass.h"
#include "HMetric.h"
#include "text_output.h"

namespace ReadingPeopleTracker
{

// definition and initialisation of static member variables
realno PCAclass::MIN_COV_ENTRY = 0.0;

PCAclass::PCAclass(char *filename, unsigned int max)
{
    ifstream data_in(filename);
    if (!data_in) 
    {
	cerror << " PCAclass::PCAclass: cannot open PCA file " << filename << " "
               << endl;
	exit(1);
    }
    
    data_in >> *this;
    max_n = max;
}

PCAclass::PCAclass(unsigned int size, unsigned int max):
    data_size(size),
    covariance(size,size,0.0),
    change_covariance(size, size, 0.0),
    mean(size),
    noise_model(),
    eigenvalues(),
    eigenvectors(),
    weights()
{
    no_processed = 0;
    max_n = max;
    mean.clear();
}

void PCAclass::process_cov(const NagVector &new_data)
{
    //  new version calculates cov(X Xt) = E(X Xt) - E(X)E(Xt)
    if (new_data.get_size() < data_size)
    {
	cerror << " PCAclass::process_cov: bad call to method " << endl;
	exit(1);
    }
    
    no_processed++;
    if (no_processed > max_n) no_processed = max_n;
    
    // update E(X)
    mean -= ((mean - new_data) / no_processed);
    
    // update E(X Xt)
    realno tmp;
    for (unsigned int i = 0; i < data_size; i++)
	for (unsigned int j = i; j < data_size; j++)
	{
	    tmp = *cov(i,j) - new_data[i] * new_data[j];
	    *cov(i,j) -= (tmp / no_processed);
	}
}

void PCAclass::process_cov(const NagVector &new_data, const NagVector &old_data)
{
    NagVector deriv;
    new_data.subtract(old_data, deriv);

    assert (new_data.get_size() >= data_size);
    
    if (new_data.get_size() < data_size)
    {
	cerror << " PCAclass::process_cov: data sizes do not match " << endl;
	exit(1);
    }
    
    no_processed++;
    if (no_processed > max_n) no_processed = max_n;
    
    // update E(X)
    mean -= ((mean - new_data) / no_processed);
    
    // update E(X Xt), E(del_X del_X_t)
    
    realno tmp;
    for (unsigned int i = 0; i < data_size; i++)
	for (unsigned int j = i; j < data_size; j++)
	{
	    tmp = *cov(i,j) - new_data[i] * new_data[j];
	    *cov(i,j) -= (tmp / no_processed);
	    tmp = change_covariance.read(i,j) - deriv[i] * deriv[j];
	    *change_covariance.get(i,j) -= (tmp / no_processed);
	}
    
}
void PCAclass::restore_covariance()
{
    // subtract off E(X)E(Xt) (see above)
    unsigned int i,j;
    for (i = 0; i < data_size; i++)
	for (j = i; j < data_size; j++)
	{
	    *cov(i,j) -= mean[i] * mean[j];
	    *cov(j,i) = *cov(i, j);
	}
    for (i = 0; i < data_size; i++)
	for (j = i; j < data_size; j++)
	{
	    realno rnk_cor = ((*cov(i,j)) * (*cov(i,j))) / 
		((*cov(i,i)) * (*cov(j,j)));
	    if (rnk_cor < MIN_COV_ENTRY)
	    {
		*cov(i,i) -= *cov(i,j);
		*cov(j,j) -= *cov(i,j);
		*cov(i,j) = 0;
		*cov(j,i) = 0;
		
	    }
	}
}

void PCAclass::process_mean(NagVector &new_data)
{
    no_processed++;
    if (no_processed > max_n)
	no_processed = max_n;

    mean.subtract((mean - new_data) / no_processed, mean);
}


void PCAclass::get_evals()
{
//  restore_covariance();
//  covariance.get_eigenvectors(eigenvectors, eigenvalues);
//  eigenvalues.output();
//  break_down_evals(eigenvalues);
	HMetric::setup_H_metric(data_size);
    
// NagMatrix tmp
// covariance.ortho_transform(*get_H_root(), tmp);
    
    NagMatrix tmp;
    covariance.get_eigenvectors(tmp,eigenvalues);
    
    HMetric::get_H_inv_root()->multiply(tmp, eigenvectors);
    NagVector tmp_v;
    mean.copy(tmp_v);
    HMetric::get_H_inv_root()->multiply(tmp_v, mean);
    
//  break_down_evals(eigenvalues);
    
}

void PCAclass::add_noise(realno noise_variance, unsigned int no_spline_pnts)
{
    
//  for (unsigned int i = 0; i < data_size; i++)
//    *covariance.get(i,i) += noise_variance;
    
    
    HMetric::setup_H_metric(data_size);
    NagMatrix tmp1;
    HMetric::build_covariance_matrix(tmp1, data_size / 2, no_spline_pnts);
    NagMatrix delta_cov;
    tmp1.ortho_transform(*HMetric::get_H_root(), delta_cov, false);
    delta_cov.scale(noise_variance, delta_cov);
    covariance.add(delta_cov, covariance);
    
    
    NagMatrix tmp;
    
    covariance.get_eigenvectors(tmp,eigenvalues);
    
    HMetric::get_H_inv_root()->multiply(tmp, eigenvectors); 
}

void PCAclass::get_noise_model()
{
    NagMatrix tmp;
    if (HMetric::get_H_root()->get_data() != NULL)
	HMetric::get_H_root()->multiply(eigenvectors, tmp);
    else eigenvectors.copy(tmp);
    change_covariance.ortho_transform(tmp, change_covariance);
    change_covariance.read_diagonal(noise_model);
//  noise_model.output();
}

void PCAclass::get_mode(unsigned int index, realno lmda, NagVector &result)
{
    NagVector e_i(data_size, get_evec(index));
    mean.add(lmda * e_i, result);
    e_i.reset();
}

void PCAclass::adjust_mode(unsigned int index, realno lmda, NagVector &result)
{
    NagVector e_i(data_size, get_evec(index));
    result.add(lmda * e_i, result);
    e_i.reset();
}

realno PCAclass::Hdot(const NagVector &v1, const NagVector &v2)
{
    HMetric::setup_H_metric(data_size);
    NagVector tmp;
    HMetric::get_H()->multiply(v2, tmp);
    return v1.dot(tmp);
}

void PCAclass::map_to_shape(unsigned int no_modes, NagVector &result, realno
			    max_distance)
{
    unsigned int i;
    NagVector lambda(no_modes);
    realno m_dist = 0;
    for (i = 0; i < no_modes; i++)
    {
	NagVector e_i(data_size, get_evec(i+1));
	lambda[i] = Hdot(e_i, (result - mean));
	m_dist += (lambda[i] * lambda[i]) / get_eval(i+1);
	e_i.reset();
    }
    if (m_dist >= max_distance)
	lambda.scale(sqrt(max_distance / m_dist), lambda);
    
    get_mode(1,lambda[0], result);
    for (i = 1; i < no_modes; i++)
	adjust_mode(i+1, lambda[i], result);
}

void PCAclass::map_to_model(unsigned int no_modes, NagVector &shape, NagVector &result)
{
    if (shape.get_size() < data_size)
    {
	// data only containts first 'n' components
	// use pseudo-inverse to find shape parameters
	
	// solve Pb = dx
	NagMatrix P(shape.get_size(), no_modes);
	for (unsigned int i = 0; i < no_modes; i++)
	{
	    NagVector e_i(shape.get_size(), get_evec(i+1));
	    P.set_column(i, e_i);
	    e_i.reset();
	}
	
	NagMatrix P_inv;
	
	P.pseudo_invert(P_inv);
	P_inv.multiply((shape - mean), result);
    }
    else
    {
	if (result.get_data() == NULL)
	    result.reconstruct(no_modes);
	
	for (unsigned int i = 0; i < no_modes; i++)
	{
	    NagVector e_i(data_size, get_evec(i+1));
	    result[i] = Hdot((shape - mean), e_i);
	    e_i.reset();
	}
    }
}


realno PCAclass::map_to_mode(unsigned int index, NagVector &shape)
{
    NagVector e_i(data_size, get_evec(index+1));
    NagVector diff(shape - mean);
    realno res = Hdot(diff, e_i);
    e_i.reset();
    return res;
}


ostream &operator<<(ostream &strm_out, PCAclass &model)
{
    
    strm_out << "PCA_MODEL" << endl;
    strm_out << " no_processed " << model.no_processed << endl;
//    unsigned int data_size = model.data_size;
    strm_out << " data_size " << model.data_size << endl;
    strm_out << " Covariance_Matrix " << model.covariance;
    strm_out << " Mean " << model.mean;
    strm_out << " Weights " << model.weights;
    strm_out << " Eigenvalues " << model.eigenvalues;
    strm_out << " Eigenvectors " << model.eigenvectors;
    if (model.noise_model.get_data() != NULL)
	strm_out << " Noise_Model " << model.noise_model;
    return strm_out;
}

istream &operator>>(istream &strm_in, PCAclass &model)
{
    char dummy[256];

    strm_in >> dummy;
    strm_in >> dummy >> model.no_processed >> dummy >> model.data_size;
    strm_in >> dummy >> model.covariance;
    strm_in >> dummy >> model.mean;
    strm_in >> dummy >> model.weights;
    strm_in >> dummy >> model.eigenvalues;
    strm_in >> dummy >> model.eigenvectors;
    strm_in >> dummy;

    if (strcmp(dummy,"Noise_Model") == 0)	
	strm_in >> model.noise_model;
    else
	model.eigenvalues.copy(model.noise_model);

    return strm_in;
}

} // namespace ReadingPeopleTracker
