///////////////////////////////////////////////////////////////////////////////
//                                                                           //
//  feature_track.cc (structure copied from simple_track.cc)                 //
//                                                                           //
//  This tracker tracks human features such as head etc                      //
//    using the SkeletonTracker class (qv)                                   //
//                                                                           //
//  Author    : Nils T Siebel (nts)                                          //
//  Created   : Fri Apr 06 15:14:36 BST 2001                                 //
//  Revision  : 0.0 of Fri Apr 06 15:14:36 BST 2001                          //
//  Copyright : The University of Reading                                    //
//                                                                           //
//  Changes:                                                                 //
//    nts: rev 1.0: initial working revision    The Future                   //
//                                                                           //
///////////////////////////////////////////////////////////////////////////////


static const char *skeleton_track_Revision = "@(#) feature_track.cc, rev 0.0 of Fri Apr 06 15:14:36 BST 2001, Author Nils T Siebel, Copyright (c) 2001 The University of Reading";

#include "Image.h"
#include <stdio.h>
#include <getopt.h>
#ifndef NO_DISPLAY
#include <unistd.h>   // for sleep()
#endif   // #ifndef NO_DISPLAY
#include "TrackedProfileSet.h"

#include "MovieStore.h"

#include "imgsrc_utils.h"

#include "Skeleton.h"
#include "SkeletonTracker.h"

extern bool Skeleton::draw_colour;
extern bool Skeleton::draw_origin;
extern bool Skeleton::draw_joints;
extern bool Skeleton::draw_bones;
extern bool Skeleton::draw_search_lines;

#include "HumanFeatures.h"

extern bool HumanFeatures::draw_colour;
extern bool HumanFeatures::draw_head_search_area;
extern bool HumanFeatures::draw_head_bbox;
extern bool HumanFeatures::draw_head_centre;
extern bool HumanFeatures::draw_shoulder_width;

#include "icons.h"

#define USAGE "skeleton_track -i options [-p file] [-l n] \
[-q] [-o file] \n\
-i input_options (-i help for details) \n\
-p parameter_file \n\
-l last frame number to process \n\
-o output numbered jpeg files, give first name \n\
-q quiet mode -- display no images and no interface \n"


int debug_level =
Configuration.register_int("DEBUG_LEVEL", 0,
		     &debug_level, false,
		     "main",
		     "General debug level");

int MAX_OBJECTS = 
Configuration.register_int("MAX_OBJECTS", 8,
		     &MAX_OBJECTS,  false,
		     "main", 
		     "The maximum number of skeletons to track");


int main(int argc, char** argv)
{    
    ImageSource *input_source = NULL;
    bool show_images = true;
    char *output_movie = NULL;
    char *parameter_file = NULL;
    MovieStore* mv_out = NULL; 

    int last_frame = 0;
    int c;
    
    while ((c = getopt(argc, argv, "i:p:o:l:q")) != EOF)
	switch(c)
	{
	case 'i':
	    input_source = get_live_or_movie_feed(optarg);
	    break;
	    
	case 'p':
	    parameter_file = optarg;
	    break;
	    
	case 'o':
	    if (optarg != NULL)
		output_movie = optarg;
	    break;
	    
	case 'q':
	    show_images = false;
	    break;
	    
	case 'l':
	    last_frame = atoi(optarg);
	    break;
	    
	case '?':
	default:
	    fprintf(stderr, USAGE);
	    exit(1);
	}
    
    Configuration.parse_parameter_file(parameter_file);

    // set default icon for windows...
    Configuration.set_icon(skeleton_track_icon_16);
    
    if (show_images)
	Configuration.create_interface(argc, argv);
    
    ImageSource *vidsrc = input_source;
    
    if (vidsrc == NULL)
    {
	cerr << " skeleton_track: Please give video input source " << endl;
	exit(1);
    }
    
    SkeletonSet curr_skeletons;
    
    SkeletonTracker *skeleton_tracker =
	new SkeletonTracker(input_source, &curr_skeletons);

    // set all draw parameters to true for now...
    HumanFeatures::draw_colour = true;
    HumanFeatures::draw_head_search_area = true;
    HumanFeatures::draw_head_bbox = true;
    HumanFeatures::draw_head_centre = true;
    HumanFeatures::draw_shoulder_width = true;
    
    // set all draw parameters to true for now...
    Skeleton::draw_colour = true;
    Skeleton::draw_origin = true;
    Skeleton::draw_joints = true;
    Skeleton::draw_bones = true;
    Skeleton::draw_search_lines = true;

    //Configuration.output_info();
    
//      printf("*** frame %6li  ---  time %6.2f sec ***\n",
//  	   vidsrc->get_frame_id(),
//  	   (float) vidsrc->get_current_timestamp() / 1000);
    
//      if (show_images)
//  	vidsrc->display();
    
    if (output_movie != NULL)
#ifdef LINUX
	mv_out = new MovieStore(MOVIESTORE_JPEG,output_movie);
#else
        mv_out = new MovieStore(output_movie, 0, true);
#endif
    
    /********************************************************************/
    /*                                                                  */
    /*                      Main Loop                                   */
    /*                                                                  */
    /********************************************************************/

    while (vidsrc->get_current() != NULL) 
    {
	printf("******************************************\n");
	printf("*** frame %6li  ---  time %6.2f sec ***\n",
	       vidsrc->get_frame_id(),
	       (float) vidsrc->get_current_timestamp() / 1000);
	if ((last_frame != 0) && (vidsrc->get_frame_id() >= last_frame))
	    break;
	
	if (show_images)
	{
	    vidsrc->display();
	    vidsrc->get_current()->draw_in_image();
	    curr_skeletons.draw();
#ifdef DEBUG
	    gflush();
#endif
	}
//  	else
//  	{
//  	    fprintf(stderr,"\r frame %d ", vidsrc->get_frame_no());
//  	    fflush(stderr);
//  	}

	skeleton_tracker->MainLoop(MAX_OBJECTS);

	printf("******************************************\n");

	if (output_movie != NULL)
	{
	    Image *video_image = vidsrc->get_current();
	    curr_skeletons.draw_in_image(video_image);
	    mv_out->add_a_frame(video_image);
	}

//  	cout << "***  frame " << vidsrc->get_frame_no()
//  	     << "  time " << vidsrc->get_current_timestamp().tv_sec << "."
//  	     << vidsrc->get_current_timestamp().tv_usec / 100000
//  	     << " sec  ***" << endl;

#ifdef DEBUG
	gflush();  // should have been done but better safe than sorry...
	sleep(2);
#endif
	
	// delete old measurements
	curr_skeletons.delete_all();
	
	vidsrc->get_next();
    }
    
    if (output_movie != NULL)
	delete mv_out;

    cout << endl;

    return 0;
}

// Local Variables:
// compile-command: "make skeleton_track |& fgrep :"
// End:
