/* test program */

#include "Image.h"
#include <stdio.h>
#include <getopt.h>
#include "TrackedProfileSet.h"

#include "MovieStore.h"

#include "Grey8Image.h"
#include "imgsrc_utils.h"
#ifdef PROFILING
#include "profiling.h"
#endif
#include "ActiveShapeTracker.h"

#include "icons.h"

#define USAGE "simple_track -i options [-p file] [-l n] \
[-q] [-o file] \n\
-i input_options (-i help for details) \n\
-p parameter_file \n\
-l last frame number to process \n\
-o output numbered jpeg files, give first name \n\
-q quick mode -- display no images \n"

int MAX_OBJECTS = 
Configuration.register_int("MAX_OBJECTS", 8,
		     &MAX_OBJECTS,  false,
		     "main", 
		     "The maximum number of silhouettes to track");


int main(int argc, char** argv)
{    
    ImageSource *input_source = NULL;
    bool show_images = true;
    char *output_movie = NULL;
    char *parameter_file = NULL;
    MovieStore* mv_out = NULL; 

    int last_frame = 0;
    int c;
    
    while ((c = getopt(argc, argv, "i:p:o:l:q")) != EOF)   // z:
	switch(c)
	{
	case 'i':
	    input_source = get_live_or_movie_feed(optarg);
	    break;
	    
	case 'p':
	    parameter_file = optarg;
	    break;
	    
	case 'o':
	    if (optarg != NULL) output_movie = optarg;
	    break;
	    
	case 'q':
	    show_images = false;
	    break;
	    
	case 'l':
	    last_frame = atoi(optarg);
	    break;
	    
	case '?':
	default:
	    fprintf(stderr, USAGE);
	    exit(1);
	}
    
    Configuration.parse_parameter_file(parameter_file);
    
    // set default icon for windows...
    Configuration.set_icon(simple_track_icon_16);
    
    if (show_images)
	Configuration.create_interface(argc, argv);
    
    ImageSource *vidsrc = input_source;
    
    if (vidsrc == NULL)
    {
	cerr << " simple_track: Please give video input source " << endl;
	exit(1);
    }
    
    TrackedProfileSet curr_profiles;
    
    ActiveShapeTracker *people_tracker = 
	new ActiveShapeTracker(input_source, NULL, &curr_profiles);
    
    //Configuration.output_info();
    
    if (show_images)
	vidsrc->display();
    
    if (output_movie != NULL)
#ifdef LINUX
	mv_out = new MovieStore(MOVIESTORE_JPEG,output_movie);
#else
        mv_out = new MovieStore(output_movie, 0, true);
#endif
    
    /********************************************************************/
    /*                                                                  */
    /*                      Main Loop                                   */
    /*                                                                  */
    /********************************************************************/
    
    cout << NO_CONTROL_POINTS << " control_points " << endl;

#ifdef PROFILING
    initialise_timer();  // initialise timing routine, see profiling.h
    setusecs(0);
    //unsigned long start_timer;
    //unsigned long timer = start_timer = getusecs();
#endif

    while (vidsrc->get_next() != NULL) 
    {
	if ((last_frame != 0) && (vidsrc->get_frame_id() >= last_frame))
	    break;
	
#ifdef PROFILING
	fprintf(stderr,"timer: time spent on getting this frame: %i microsec \n",
	    (int) (getusecs()));
	setusecs(0);
#endif

	people_tracker->MainLoop(MAX_OBJECTS);

#ifdef PROFILING
	fprintf(stderr,"timer: time spent on tracking: %i microsec \n",
	    (int) (getusecs()));
	setusecs(0);
#endif
	
	TrackedProfileSet filtered = 
	    curr_profiles.filter_out_short(ActiveShapeTracker::SMALL_NO_FRAMES);

#ifdef PROFILING
	fprintf(stderr,"timer: time spent on filtering profiles: %i microsec \n",
	    (int) (getusecs()));
	setusecs(0);
#endif
	
	if (show_images)
	{
	    vidsrc->display();
	    curr_profiles.draw();
	}
#ifndef DEBUG
	else
#endif
	{
	    fprintf(stderr,"\r frame %li ", vidsrc->get_frame_id());
	    fflush(stderr);
	}

#ifdef PROFILING
	fprintf(stderr,"timer: time spent on displaying video: %i microsec \n",
		(int) (getusecs()));
	setusecs(0);
#endif	
	
	if (output_movie != NULL)
	{
	    curr_profiles.draw_in_image(vidsrc->get_current(),false);
	    mv_out->add_a_frame(vidsrc->get_current());
#ifdef PROFILING
	fprintf(stderr,"timer: time spent on writing output frame: %i microsec \n",
	    (int) (getusecs()));
	setusecs(0);
#endif
	}

#ifndef PROFILING
	printf("*** frame %4li  ---  time %6.2f sec ***\n",
	       vidsrc->get_frame_id(),
	       (float) vidsrc->get_current_timestamp() / 1000);
//  	cout << "***  frame " << vidsrc->get_frame_no()
//  	     << "  time " << vidsrc->get_current_timestamp().tv_sec << "."
//  	     << vidsrc->get_current_timestamp().tv_usec / 100000
//  	     << " sec  ***" << endl;
	cout << filtered;
#endif
	filtered.delete_all();
    }
    
    if (output_movie != NULL)
	delete mv_out;

    cout << endl;

    return 0;
}

// Local Variables:
// compile-command: "make simple_track |& fgrep :"
// End:
