/* test program */

#include "Image.h"
#include "ImageSource.h"
#include <stdio.h>
#include <getopt.h>
#include "TrackedProfileSet.h"
#ifdef LINUX
#include "MovieStore.h"
#endif
#include "Grey8Image.h"
#include "utils.h"
#include "imgsrc_utils.h"
#include "GroundPeopleTracker.h"

#include "icons.h"

#define USAGE "ground_track -i options [-p file] [-l n] [-q] \
[-o file][-z file] [-c] \n\
-i input image options (-i help for details) \n\
-p parameter file \n\
-l last frame number to process \n\
-q quick mode -- display no images \n\
-o optional output movie file \n\
-z optional output image file  \n"


void do_some_visualisation(GroundPeopleTracker*,ImageSource*, ProfileSet*);
void output_results(GroundPeopleTracker*, ImageSource*,
		    ProfileSet*, MovieStore*, char* );


int MAX_OBJECTS = 
Configuration.register_int("MAX_OBJECTS", 8,
		     &MAX_OBJECTS,  false,
		     "main", 
		     "The maximum number of silhouettes to track");

bool show_virtual_view = 
Configuration.register_bool("SHOW_VIRTUAL_VIEW",
                      false, &show_virtual_view,
                      true, "Options",  // owner is main but please use Options window
                      "Show the virtual camera view ?");


int main(int argc, char** argv)
{
    Profile::draw_colour = true;
    Profile::draw_boxes = false;
    Profile::draw_spline = true;
    Profile::draw_rectangle = false; // not useful
    Profile::draw_arrow = false;  // if true: use NO_CONTROL_POINTS - 1 control
                                  // points and the last data point for direction
    
    bool show_images = true;
    char *output_movie = NULL;
    char *parameter_file = NULL;
    char* output_image = NULL;
    ImageSource* input_source = NULL;
    MovieStore* mv_out = NULL; 
    
    int last_frame = 0;
    int c;
    while ((c = getopt(argc, argv, "i:p:l:qo:z:c")) != EOF)
	switch(c)
	{
	case 'i':
	    input_source = get_live_or_movie_feed(optarg);
	    break;
	    
	case 'p':
	    parameter_file = optarg;
	    break;
	    
	case 'l':
	    if (optarg != NULL) last_frame = atoi(optarg);
	    break;
	    
	case 'q':
	    show_images = false;
	    break;
	    
	case 'z':
	    if (optarg != NULL) 
		output_image = optarg;
	    break;
	    
	case 'o':
	    if (optarg != NULL) output_movie = optarg;
	    break;
	    
	case '?':
	default:
	    fprintf(stderr, USAGE);
	    exit(1);
	}
    
    Configuration.parse_parameter_file(parameter_file);

    // set default icon for windows...
    Configuration.set_icon(ground_track_icon_16);
    
    Configuration.create_interface(argc, argv);


    ImageSource *vidsrc = input_source;
    
    if (vidsrc == NULL)
    {
	cerr << " ground_track: Please give video input source " << endl;
	exit(1);
    }

    TrackedProfileSet curr_profiles; // pedestrian silhouettes
    
    GroundPeopleTracker* people_tracker = 
	new GroundPeopleTracker(vidsrc, NULL, &curr_profiles);
    
    if (show_images)
	vidsrc->display();
    
    if (output_movie != NULL)
#ifdef LINUX
	mv_out = new MovieStore(MOVIESTORE_JPEG, output_movie);
#else
        mv_out = new MovieStore(output_movie, 0, true);
#endif
    
    /********************************************************************/
    /*                                                                  */
    /*                      Main Loop                                   */
    /*                                                                  */
    /********************************************************************/
    
    while (vidsrc->get_next() != NULL) 
    {
	if ((last_frame != 0) && (vidsrc->get_frame_id() >= last_frame))
	    break;

	people_tracker->MainLoop(MAX_OBJECTS);

	if (show_images)  
	    do_some_visualisation(people_tracker, vidsrc, &curr_profiles);
	else
	{
	    fprintf(stderr,"\r frame %li ", vidsrc->get_frame_id());
	    fflush(stderr);
	}

	output_results(people_tracker, vidsrc, &curr_profiles, mv_out,
		       output_image);
    }

    if (mv_out != NULL)
	delete mv_out;
    
    return 0;

    /********************************************************************/
}


// do_some_visualisation
// display images etc
void do_some_visualisation(GroundPeopleTracker *people_tracker,
			   ImageSource *vidsrc,
			   ProfileSet *curr_profiles)
{
    static RGB32Image* rgb_viz_image = NULL;

    // display video image
    vidsrc->display();
    // (do_some_visualisation() is only called if show_images is true)
    // display profiles over current video image on screen
    curr_profiles->draw();
    
    if (show_virtual_view)
    {    
	Image* viz_image = people_tracker->get_viz_image();

	if (rgb_viz_image == NULL)
	{
	    // wrap an RGB32 Image around the 8-bit greyscale data
	    rgb_viz_image = new RGB32Image(viz_image->get_width(),
					   viz_image->get_height(),
					   viz_image->get_data());
	    rgb_viz_image->set_title("Camera view of computer model");
	}
	
	rgb_viz_image->display();
    }    
}


// output_results
// if required output movie visualisation.
// output silhouette images and data files
void output_results(GroundPeopleTracker *people_tracker,
		    ImageSource *vidsrc,
		    ProfileSet *curr_profiles,
		    MovieStore *mv_out,
		    char *output_filename)
{
    // dump a movie frame showing input image with superimposed contours
    if (mv_out != NULL)
    {
	people_tracker->get_tracked_profiles()->
	    draw_in_image(vidsrc->get_current(), 
			  GroundPeopleTracker::SMALL_NO_FRAMES);
	mv_out->add_a_frame(vidsrc->get_current());
    }
    
    // dump the silhouette image and the ground plane data file
    if (output_filename != NULL)
    {
	char str_buffer[200];
	int frame_no = vidsrc->get_frame_id();
	sprintf(str_buffer,"%s%.5d.pgm", output_filename, frame_no);
	people_tracker->get_viz_image()->save_pnm(str_buffer);
	
	sprintf(str_buffer,"%s%.5d.dat", output_filename, frame_no);
	ofstream data_out(str_buffer);
	for (curr_profiles->start(); curr_profiles->current_ok();
	     curr_profiles->forward())
	{
	    if (curr_profiles->get_current().get_type() == TRACKED_PROFILE)
	    {
		TrackedProfile* curr_prf = (TrackedProfile*)
		    &(curr_profiles->get_current());
		if (curr_prf->time_alive > GroundPeopleTracker::SMALL_NO_FRAMES)
		    curr_prf->dump_reading_info(data_out);
	    }
	}
	data_out.close();
	sprintf(str_buffer,"gzip -f %s%.5d.pgm %s%.5d.dat", output_filename,
		frame_no, output_filename, frame_no);
	system(str_buffer);	// compress image, data	  
    }
    else
    {
	// TDG 14/12/98
 	// if no output file, just write everything (inc. shape params) to stdout
	printf("*** frame %4li  ---  time %6.2f sec ***\n",
	       vidsrc->get_frame_id(),
	       (float) vidsrc->get_current_timestamp() / 1000);
	
//	cout << "***  frame " << vidsrc->get_frame_no() - 1
//	     << "  time " << vidsrc->get_current_timestamp().tv_sec << "."
//	     << vidsrc->get_current_timestamp().tv_usec / 100000
//	     << " sec  ***" << endl;

	for (curr_profiles->start(); curr_profiles->current_ok();
	     curr_profiles->forward())
        {
	    if (curr_profiles->get_current().get_type() == TRACKED_PROFILE)
            {
		TrackedProfile *curr_prf = (TrackedProfile*)
		    &(curr_profiles->get_current());
		if (curr_prf->time_alive > GroundPeopleTracker::SMALL_NO_FRAMES)
		    curr_prf->output_to_stream(cout);
            }
        }

	flush(cout);
    }
}

// Local Variables:
// compile-command: "make ground_track |& fgrep :"
// End:
