///////////////////////////////////////////////////////////////////////////////
//                                                                           //
//  feature_track.cc (structure copied from simple_track.cc)                 //
//                                                                           //
//  This tracker tracks human features such as head etc                      //
//    using the HumanFeatureTracker class (qv)                               //
//                                                                           //
//  Author    : Nils T Siebel (nts)                                          //
//  Created   : Fri Apr 06 15:14:36 BST 2001                                 //
//  Revision  : 0.0 of Fri Apr 06 15:14:36 BST 2001                          //
//  Copyright : The University of Reading                                    //
//                                                                           //
//  Changes:                                                                 //
//    nts: rev 1.0: initial working revision    The Future                   //
//                                                                           //
///////////////////////////////////////////////////////////////////////////////


static const char *feature_track_Revision = "@(#) feature_track.cc, rev 0.0 of Fri Apr 06 15:14:36 BST 2001, Author Nils T Siebel, Copyright (c) 2001 The University of Reading";

#include "Image.h"
#include <stdio.h>
#include <getopt.h>
#ifndef NO_DISPLAY
#include <unistd.h>   // for sleep()
#endif   // #ifndef NO_DISPLAY

#include "TrackedProfileSet.h"

#include "MovieStore.h"

#include "imgsrc_utils.h"
#include "utils.h"

#include "HumanFeatures.h"
#include "HumanFeatureTracker.h"

extern bool HumanFeatures::draw_colour;
extern bool HumanFeatures::draw_head_search_area;
extern bool HumanFeatures::draw_head_bbox;
extern bool HumanFeatures::draw_head_centre;
extern bool HumanFeatures::draw_shoulder_width;

#include "icons.h"

#define USAGE "feature_track -i options [-p file] [-l n] \
[-q] [-o file] \n\
-i input_options (-i help for details) \n\
-p parameter_file \n\
-l last frame number to process \n\
-o output numbered jpeg files, give first name \n\
-q quiet mode -- display no images and no interface \n"

int debug_level =
Configuration.register_int("DEBUG_LEVEL", 0,
		     &debug_level, false,
		     "main",
		     "General debug level");

int MAX_OBJECTS = 
Configuration.register_int("MAX_OBJECTS", 8,
		     &MAX_OBJECTS,  false,
		     "main", 
		     "The maximum number of human feature sets to track");


int main(int argc, char** argv)
{    
    ImageSource *input_source = NULL;
    bool show_images = true;
    char *output_movie = NULL;
    char *parameter_file = NULL;
    MovieStore* mv_out = NULL; 

    int last_frame = 0;
    int c;
    
    while ((c = getopt(argc, argv, "i:p:o:l:q")) != EOF)
	switch(c)
	{
	case 'i':
	    input_source = get_live_or_movie_feed(optarg);
	    break;
	    
	case 'p':
	    parameter_file = optarg;
	    break;
	    
	case 'o':
	    if (optarg != NULL)
		output_movie = optarg;
	    break;
	    
	case 'q':
	    show_images = false;
	    break;
	    
	case 'l':
	    last_frame = atoi(optarg);
	    break;
	    
	case '?':
	default:
	    fprintf(stderr, USAGE);
	    exit(1);
	}
    
    Configuration.parse_parameter_file(parameter_file);

    if (show_images)
    {
	// set default icon for windows...
	Configuration.set_icon(feature_track_icon_16);
	Configuration.create_interface(argc, argv);
    }
    
    ImageSource *vidsrc = input_source;
    
    if (vidsrc == NULL)
    {
	cerr << " feature_track: Please give video input source " << endl;
	exit(1);
    }
    
    HumanFeatureTracker *feature_tracker = new HumanFeatureTracker(input_source);

    
    // set all draw parameters to true for now...
    HumanFeatures::draw_colour = true;
    HumanFeatures::draw_head_search_area = true;
    HumanFeatures::draw_head_bbox = true;
    HumanFeatures::draw_head_centre = true;
    HumanFeatures::draw_shoulder_width = true;
    
    //Configuration.output_info();
    
    printf("*** frame %4li  ---  time %6.2f sec ***\n",
	   vidsrc->get_frame_id(),
	   (float) vidsrc->get_current_timestamp() /1000);
    
    if (show_images)
	vidsrc->display();
    
    if (output_movie != NULL)
#ifdef LINUX
	mv_out = new MovieStore(MOVIESTORE_JPEG,output_movie);
#else
        mv_out = new MovieStore(output_movie, 0, true);
#endif
    
    /********************************************************************/
    /*                                                                  */
    /*                      Main Loop                                   */
    /*                                                                  */
    /********************************************************************/

    while (vidsrc->get_next() != NULL) 
    {
	if ((last_frame != 0) && (vidsrc->get_frame_id() >= last_frame))
	    break;
	
	if (show_images)
	{
	    vidsrc->display();
	    // curr_profiles.draw();
	}
//  	else
//  	{
//  	    fprintf(stderr,"\r frame %d ", vidsrc->get_frame_no());
//  	    fflush(stderr);
//  	}

	feature_tracker->MainLoop(MAX_OBJECTS);

	if (output_movie != NULL)
	{
	    // curr_profiles.draw_in_image(vidsrc->get_current(),
	    //			ActiveShapeTracker::SMALL_NO_FRAMES);
	    mv_out->add_a_frame(vidsrc->get_current());
	}

	printf("*** frame %6li  ---  time %6.2f sec ***\n",
	       vidsrc->get_frame_id(),
	       (float) vidsrc->get_current_timestamp() / 1000);
//  	cout << "***  frame " << vidsrc->get_frame_no()
//  	     << "  time " << vidsrc->get_current_timestamp().tv_sec << "."
//  	     << vidsrc->get_current_timestamp().tv_usec / 100000
//  	     << " sec  ***" << endl;

#ifdef DEBUG
	gflush();  // should have been done but better safe than sorry...
	sleep(1);
#endif

    }
    
    if (output_movie != NULL)
	delete mv_out;

    cout << endl;

    return 0;
}

// Local Variables:
// compile-command: "make feature_track"
// End:
